from bs4 import BeautifulSoup
import json
import pandas as pd
import requests


class WebScrapper:
    PAGE_URL = ''
    TYPES = []

    def __init__(self):
        self.content = self.fetch_page_content()
        self.data = None
        self.header = None

    def get_data(self, serialize=True):
        if self.content is None:
            return ''

        self.process_page()
        return self.serialize_data() if serialize is True else self.data

    def extract_header(self):
        pass

    def extract_content(self, page):
        pass

    def extract_table(self):
        pass

    def fetch_page_content(self):
        page = requests.get(self.PAGE_URL)
        return self.extract_content(page) if page.status_code == 200 else None

    def filter_data(self):
        pass

    def parse_row(self, row):
        return [self.parse_tag(tag, dtype) for tag, dtype in zip(row.find_all('td'), self.TYPES)]

    def parse_tag(self, tag, dtype=str):
        return dtype(tag.text.strip())

    def process_page(self):
        rows, labels = self.extract_table()
        rows = list(map(self.parse_row, rows))
        labels = list(map(self.parse_tag, labels))

        self.data = pd.DataFrame(rows, columns=labels)
        self.filter_data()

    def serialize_data(self):
        data_json = {
            'data': json.dumps(self.data.to_dict('records')),
            'meta': json.dumps(self.header)
        }
        return json.dumps(data_json)


class StockScrapper(WebScrapper):
    PAGE_URL = 'https://www.money.pl/gielda/indeksy_gpw/wig/'
    TYPES = [str, int, float, float, float, float, float]

    def extract_content(self, page):
        soup = BeautifulSoup(page.text, 'html.parser')
        content = soup.find('div', {'id': 'indeks_stock_table'})
        return content

    def extract_header(self):
        self.header = self.content.find('div', {'class': 'hd inbox'}).text

    def extract_table(self):
        table = self.content.find('table', {'class', 'tabela big m0 tlo_biel'})
        labels = table.find('thead').find('tr').find_all('th')
        rows = table.find('tbody').find_all('tr')
        return rows, labels

    def filter_data(self):
        self.data = self.data[['Nazwa  spółki', 'Pakiet', 'Zmiana kursu spółki [%]', 'Kurs [PLN]']]
        self.data.columns = ['name', 'packet', 'change', 'course']

    def parse_tag(self, tag, dtype=str):
        return dtype(tag.text.strip().replace(',', '.'))


class CurrencyScrapper(WebScrapper):
    PAGE_URL = 'https://www.money.pl/pieniadze/nbp/srednie/'
    TYPES = [str, str, str, float, float]

    def extract_content(self, page):
        soup = BeautifulSoup(page.text, 'html.parser')
        content = soup.find('div', {'id': 'gielda'})
        return content

    def extract_header(self):
        self.header = self.content.find('div', {'class': 'hd1'}).text

    def extract_table(self):
        table = self.content.find('table', {'class', 'tabela big m0 tlo_biel'})
        labels = table.find('thead').find('tr').find_all('th')
        rows = table.find('tbody').find_all('tr')
        return rows, labels

    def filter_data(self):
        self.data = self.data[['Nazwa', 'Symb. waluty', 'Kurs średni', 'Zmiana*']]
        self.data.columns = ['name', 'code', 'course', 'change']

    def parse_tag(self, tag, dtype=str):
        return dtype(tag.text.strip().replace(',', '.').replace('%', ''))


class CryptoScrapper(WebScrapper):
    PAGE_URL = 'http://www.cryptocurrencyprices.net/'
    TYPES = [str, str, str, int, float, float, float, str, float]

    def extract_content(self, page):
        soup = BeautifulSoup(page.text, 'html.parser')
        table = soup.select_one('body > table > tr:nth-of-type(2) > td:nth-of-type(2) > table > tr:nth-of-type(3) > td > table:nth-of-type(2)')
        return table

    def extract_header(self):
        self.header = 'Bieżące zestawienie najpopularniejszych kryptowalut'

    def extract_table(self):
        labels = self.content.select_one('tr').find_all('td')
        rows = self.content.find_all('tr')[1:]
        return rows, labels

    def filter_data(self):
        self.data = self.data[['Currency', 'Symbol', 'Market Cap', 'Price', 'Circulation']]
        self.data.columns = ['name', 'code', 'market', 'price', 'circulation']

    def parse_tag(self, tag, dtype=str):
        return dtype(tag.text.strip().replace(',', '').replace('%', '').replace('$', '').replace('?', '0'))
